/*:
 * @plugindesc Customizes inventory categories and ensures items use their database-defined effects, including "Relationships" and "Options" categories. 
 * @author Clay Davis
 * @target MZ
 * 
 * @help 
 * This plugin customizes inventory categories in the Item menu based on Note tags.
 *
 * How to Use:
 * - Add one of the following tags to the Note field of an item:
 *   <items> - Adds the item to the "Items" category.
 *   <key items> - Adds the item to the "Key Items" category.
 *   <relationships> - Adds the item to the "Relationships" category.
 *   <options> - Adds the item to the "Options" category.
 *
 * Behavior:
 * - Items fully rely on their database-defined effects (e.g., common events).
 * - Item descriptions are displayed in the help window at the bottom of the screen.
 */

(() => {
    // Define the custom categories
    const customCategories = [
        { symbol: 'items', name: 'Items' },
        { symbol: 'keyItems', name: 'Key Items' },
        { symbol: 'relationships', name: 'Relationships' },
        { symbol: 'options', name: 'Options' }
    ];

    // Override the Window_ItemCategory to use custom categories
    Window_ItemCategory.prototype.makeCommandList = function() {
        for (const category of customCategories) {
            this.addCommand(category.name, category.symbol);
        }
    };

    // Override the Window_ItemList to control quantity display
    const _Window_ItemList_drawItemNumber = Window_ItemList.prototype.drawItemNumber;
    Window_ItemList.prototype.drawItemNumber = function(item, x, y, width) {
        // Do not display item quantity for specific categories
        if (['keyItems', 'relationships', 'options'].includes(this._category)) {
            return;
        }
        // Default behavior for other categories
        _Window_ItemList_drawItemNumber.call(this, item, x, y, width);
    };

    // Create the help window for item descriptions
    Scene_Item.prototype.createHelpWindow = function() {
        const rect = this.helpWindowRect();
        this._helpWindow = new Window_Help(rect);
        this.addWindow(this._helpWindow);
    };

    Scene_Item.prototype.helpWindowRect = function() {
        const wx = 0;
        const wy = Graphics.boxHeight - Window_Help.prototype.fittingHeight(2); // Positioned at the bottom
        const ww = Graphics.boxWidth;
        const wh = Window_Help.prototype.fittingHeight(2); // Height for two lines
        return new Rectangle(wx, wy, ww, wh);
    };

    // Update the logic for determining which items are included in each category
    Scene_Item.prototype.createItemWindow = function() {
        const rect = this.itemWindowRect();
        this._itemWindow = new Window_ItemList(rect);
        this._itemWindow.setHelpWindow(this._helpWindow); // Link the help window
        this._itemWindow.setHandler('ok', this.onItemOk.bind(this));
        this._itemWindow.setHandler('cancel', this.onItemCancel.bind(this));
        this._categoryWindow.setItemWindow(this._itemWindow);

        this._itemWindow.setCategory = function(category) {
            this._category = category;
            this.refresh();
        };

        this._itemWindow.includes = function(item) {
            if (!item) return false;

            const notes = item.note.toLowerCase(); // Convert notes to lowercase
            switch (this._category) {
                case 'items':
                    return DataManager.isItem(item) && notes.includes('<items>');
                case 'keyItems':
                    return DataManager.isItem(item) && notes.includes('<key items>');
                case 'relationships':
                    return DataManager.isItem(item) && notes.includes('<relationships>');
                case 'options':
                    return DataManager.isItem(item) && notes.includes('<options>');
                default:
                    return false;
            }
        };

        this._itemWindow.refresh();
        this.addWindow(this._itemWindow);
    };

    // Restore default item behavior for triggering effects
    Scene_Item.prototype.onItemOk = function() {
        const item = this.item();
        if (!item) {
            this.onItemCancel();
            return;
        }

        // Trigger the item's effects as defined in the database
        $gameParty.setLastItem(item);
        this.useItem(); // Use the item, which triggers its common event or effect
        this.activateItemWindow(); // Reactivate the item window after usage
    };

    Scene_Item.prototype.onItemCancel = function() {
        this._itemWindow.deselect();
        this._categoryWindow.activate();
    };

    // Adjust item window layout to leave space for the help window
    Scene_Item.prototype.itemWindowRect = function() {
        const wx = 0;
        const wy = this._categoryWindow.height;
        const ww = Graphics.boxWidth;
        const wh = Graphics.boxHeight - this._categoryWindow.height - this._helpWindow.height;
        return new Rectangle(wx, wy, ww, wh);
    };
})();